<?php
/**
 * The MIT License (MIT)
 *
 * @Author: sharky72 (https://github.com/KocourKuba)
 * Original code from DUNE HD
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

require_once 'json_serializer.php';

/**
 * @property string|null $screen_id // Screen ID
 * @property string|null $channel_id // Channel ID
 * @property string|null $group_id // Group ID
 * @property string|null $row_id // row id (used in NewUI)
 * @property string|null $save_data // parameter used to save data
 * @property string|null $id // ID
 * @property bool|null $is_favorite // Is selected media url point to the favorite folder
 * @property int|mixed|null $archive_tm // timestamp of the archive position playback, -1 live broadcast
 * @property string|null $no_internet // is not valid EPS screen
 * @property string|null movie_id // Movie ID
 * @property string|null category_id // Movie Category ID
 * @property string|null season_id // Season ID
 * @property string|null episode_id // Episode ID
 * @property string|null genre_id // Movie Genre ID
 * @property string|null name // search name
 * @property string|null return_index // return index
 */
class MediaURL extends Json_Serializer
{
    // Original media-url string.
    /**
     * @var mixed|null
     */
    protected $str;

    // If media-url string contains map, it's decoded here.
    // Null otherwise.
    protected $map;

    ///////////////////////////////////////////////////////////////////////

    /**
     * @param string $str
     * @param array|null $map
     */
    private function __construct($str, $map)
    {
        $this->str = $str;
        $this->map = $map;
    }

    ///////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////

    /**
     * @param array $m
     * @param bool $raw_encode
     * @return MediaURL
     */
    public static function make($m, $raw_encode = false)
    {
        return self::decode(self::encode($m, $raw_encode));
    }

    /**
     * @param string $s
     * @return MediaURL
     */
    public static function decode($s = '')
    {
        if (strpos($s, '{') !== 0) {
            return new MediaURL($s, null);
        }

        return new MediaURL($s, json_decode($s));
    }

    /**
     * @param array $m
     * @param bool $raw_encode
     * @return false|string
     */
    public static function encode($m, $raw_encode = false)
    {
        return $raw_encode ? pretty_json_format($m) : json_encode($m);
    }

    /**
     * @param mixed $key
     */
    public function __unset($key)
    {
        if (is_null($this->map)) {
            return;
        }

        unset($this->map->{$key});
    }

    /**
     * @param mixed $key
     * @return mixed|null
     */
    public function __get($key)
    {
        if (is_null($this->map)) {
            return null;
        }

        return isset($this->map->{$key}) ? $this->map->{$key} : null;
    }

    ///////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////

    /**
     * @param mixed $key
     * @param mixed $value
     */
    public function __set($key, $value)
    {
        if (is_null($this->map)) {
            $this->map = (object)array();
        }

        $this->map->{$key} = $value;
    }

    /**
     * @param mixed $key
     * @return bool
     */
    public function __isset($key)
    {
        if (is_null($this->map)) {
            return false;
        }

        return isset($this->map->{$key});
    }

    ///////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////

    /**
     * @return string
     */
    public function __toString()
    {
        return (string)MediaURL::encode($this->map, true);
    }

    ///////////////////////////////////////////////////////////////////////

    /**
     * @return MediaURL
     */
    public function duplicate()
    {
        return clone $this;
    }


    /**
     * @return string
     */
    public function get_raw_string()
    {
        return $this->str;
    }

    /**
     * @param bool $raw_encode
     * @return string
     */
    public function get_media_url_string($raw_encode = false)
    {
        return MediaURL::encode($this->map, $raw_encode);
    }
}
